import SwiftUI
import Cocoa
import Carbon.HIToolbox

// ================================================================
// MARK: - App Entry
// ================================================================
@main
struct InputLanguageNotifierApp: App {
    @NSApplicationDelegateAdaptor(AppDelegate.self) var appDelegate
    var body: some Scene { Settings { EmptyView() } }
}

// ================================================================
// MARK: - HUD Manager (영구 윈도우 재사용 + 현재 디스플레이 중앙 배치 + 세로가운데 + 자동축소)
// ================================================================
final class HUDManager {
    static let shared = HUDManager()

    private var window: NSWindow!
    private var label: NSTextField!
    private var bg: NSVisualEffectView!
    private var hideTimer: DispatchSourceTimer?

    // 높이는 고정, 가로는 텍스트/폰트에 따라 가변
    private let baseHeight: CGFloat = 90
    private let minWidth: CGFloat = 160
    private let maxWidth: CGFloat = 520   // 살짝 넉넉하게
    private let paddingX: CGFloat = 24
    private let paddingY: CGFloat = 18

    private init() {
        // 초기 윈도우 (위치는 show() 때마다 재배치)
        let rect = NSRect(x: 0, y: 0, width: minWidth, height: baseHeight)
        let w = NSWindow(contentRect: rect, styleMask: [.borderless], backing: .buffered, defer: false)
        w.level = .statusBar
        w.isOpaque = false
        w.backgroundColor = .clear
        w.ignoresMouseEvents = true
        w.collectionBehavior = [.canJoinAllSpaces, .stationary]
        w.acceptsMouseMovedEvents = false
        w.isReleasedWhenClosed = false
        w.orderOut(nil)

        // 컨텐츠 뷰 + Auto Layout
        let contentView = NSView(frame: rect)
        contentView.translatesAutoresizingMaskIntoConstraints = false

        let bg = NSVisualEffectView(frame: contentView.bounds)
        bg.material = .hudWindow
        bg.state = .active
        bg.wantsLayer = true
        bg.layer?.cornerRadius = 14
        bg.translatesAutoresizingMaskIntoConstraints = false

        let label = NSTextField(labelWithString: "")
        label.identifier = .init("HUDLabel")
        label.font = .systemFont(ofSize: 30, weight: .semibold) // 기본 폰트 (필요시 축소)
        label.textColor = .white
        label.alignment = .center
        label.lineBreakMode = .byClipping // 잘라내기 (축소 후에도 넘치면 마지막 방어)
        label.translatesAutoresizingMaskIntoConstraints = false

        contentView.addSubview(bg)
        contentView.addSubview(label)
        w.contentView = contentView

        // Auto Layout 제약: 배경은 꽉 채우고, 라벨은 정중앙
        NSLayoutConstraint.activate([
            // 배경 꽉 채우기
            bg.leadingAnchor.constraint(equalTo: contentView.leadingAnchor),
            bg.trailingAnchor.constraint(equalTo: contentView.trailingAnchor),
            bg.topAnchor.constraint(equalTo: contentView.topAnchor),
            bg.bottomAnchor.constraint(equalTo: contentView.bottomAnchor),

            // 라벨은 세로/가로 정중앙 + 좌우/상하 패딩
            label.centerXAnchor.constraint(equalTo: contentView.centerXAnchor),
            label.centerYAnchor.constraint(equalTo: contentView.centerYAnchor),
            label.leadingAnchor.constraint(greaterThanOrEqualTo: contentView.leadingAnchor, constant: paddingX),
            label.trailingAnchor.constraint(lessThanOrEqualTo: contentView.trailingAnchor, constant: -paddingX),
            label.topAnchor.constraint(greaterThanOrEqualTo: contentView.topAnchor, constant: paddingY),
            label.bottomAnchor.constraint(lessThanOrEqualTo: contentView.bottomAnchor, constant: -paddingY)
        ])

        self.window = w
        self.label = label
        self.bg = bg
    }

    /// 현재 디스플레이(마우스 커서가 있는 화면)
    private func currentScreen() -> NSScreen {
        let location = NSEvent.mouseLocation
        for screen in NSScreen.screens where screen.frame.contains(location) { return screen }
        return NSScreen.main ?? NSScreen.screens.first!
    }

    /// 주어진 화면 중앙 프레임
    private func centeredRect(for screen: NSScreen, size: NSSize) -> NSRect {
        let f = screen.frame
        return NSRect(
            x: f.midX - size.width/2,
            y: f.midY - size.height/2,
            width: size.width, height: size.height
        )
    }

    /// 문자열 폭 측정
    private func textWidth(_ text: String, font: NSFont) -> CGFloat {
        (text as NSString).size(withAttributes: [.font: font]).width
    }

    /// 텍스트가 maxWidth 안에 들어가도록 폰트 크기를 줄여서 반환 (최소 18pt)
    private func fittingFont(for text: String, base: NSFont, maxWidth: CGFloat) -> NSFont {
        var size = base.pointSize
        var font = base
        var width = textWidth(text, font: font) + paddingX * 2
        while width > maxWidth && size > 18 {
            size -= 1
            font = NSFont.systemFont(ofSize: size, weight: .semibold)
            width = textWidth(text, font: font) + paddingX * 2
        }
        return font
    }

    /// 텍스트 길이에 맞춘 HUD 가로폭 계산
    private func widthFor(text: String, font: NSFont) -> CGFloat {
        let w = textWidth(text, font: font) + paddingX * 2
        return max(minWidth, min(maxWidth, ceil(w)))
    }

    func show(text: String, duration: TimeInterval = 1.2) {
        if !Thread.isMainThread { DispatchQueue.main.async { self.show(text: text, duration: duration) }; return }

        // 1) 폰트 자동 축소 적용
        let baseFont = label.font ?? .systemFont(ofSize: 30, weight: .semibold)
        let fittedFont = fittingFont(for: text, base: baseFont, maxWidth: maxWidth)
        label.font = fittedFont
        label.stringValue = text

        // 2) 사이즈 계산 + 현재 디스플레이 중앙 배치
        let width = widthFor(text: text, font: fittedFont)
        let size = NSSize(width: width, height: baseHeight)
        let screen = currentScreen()
        let frame = centeredRect(for: screen, size: size)
        window.setFrame(frame, display: true)

        // 3) 표시
        window.alphaValue = 1.0
        window.orderFrontRegardless()

        // 4) 타이머 재설정
        hideTimer?.cancel()
        hideTimer = nil
        let timer = DispatchSource.makeTimerSource(queue: .main)
        timer.schedule(deadline: .now() + duration)
        timer.setEventHandler { [weak self] in
            self?.window.orderOut(nil)  // 닫지 않고 숨김
        }
        timer.activate()
        hideTimer = timer
    }

    func invalidate() {
        if !Thread.isMainThread { DispatchQueue.main.async { self.invalidate() }; return }
        hideTimer?.cancel()
        hideTimer = nil
        window.orderOut(nil)
    }
}

// ================================================================
// MARK: - AppDelegate
// ================================================================
final class AppDelegate: NSObject, NSApplicationDelegate {
    private var statusItem: NSStatusItem?
    private var currentInputSource: String = "알 수 없음"

    // “앱 전환 → 입력소스 변경” 필터
    private var lastActivatedAt: Date = .distantPast
    private let appSwitchThresholdMs: Double = 600
    private var appObserver: NSObjectProtocol?
    private var distToken: NSObjectProtocol?
    
    // 알림 모드 설정: true = 항상 알림, false = 앱 전환 시에만 알림
    private let notifyAlways: Bool = true

    func applicationDidFinishLaunching(_ notification: Notification) {
        setupStatusItem()
        currentInputSource = getCurrentInputSource()
        updateStatusMenu()

        // 앱 활성화 감지(전환 시각 기록)
        appObserver = NSWorkspace.shared.notificationCenter.addObserver(
            forName: NSWorkspace.didActivateApplicationNotification,
            object: nil, queue: .main
        ) { [weak self] _ in
            self?.lastActivatedAt = Date()
        }

        // 입력 소스 변경 알림 (블록 기반, 메인 큐)
        distToken = DistributedNotificationCenter.default().addObserver(
            forName: NSNotification.Name(kTISNotifySelectedKeyboardInputSourceChanged as String),
            object: nil, queue: .main
        ) { [weak self] _ in
            self?.handleInputSourceChange()
        }

        NSLog("입력 언어 알림 앱 시작. 현재: \(currentInputSource)")
    }

    func applicationWillTerminate(_ notification: Notification) {
        cleanup()
    }

    deinit { cleanup() }

    /// 입력 소스 문자열을 메뉴바 배지("A" 또는 "한")로 변환
    private func shortBadge(for input: String) -> String {
        let s = input.lowercased()

        // 한글판별: 이름 또는 ID에 korean/hangul/2setkorean 포함
        if s.contains("korean") || s.contains("hangul") || s.contains("2setkorean") || s.contains("한글") || s.contains("한국") {
            return "한"
        }

        // 영문판별: ABC / US / U.S. / English 등
        if s.contains("abc") || s.contains("us") || s.contains("u.s.") || s.contains("english") || s.contains("en") {
            return "A"
        }

        // 그 외(중국어/일본어 등)에는 첫 글자(대문자) 정도로 표시
        if let first = input.trimmingCharacters(in: .whitespacesAndNewlines).first {
            return String(first).uppercased()
        }
        return "A"
    }
    
    // -------------------- Status Item --------------------
    private func setupStatusItem() {
        statusItem = NSStatusBar.system.statusItem(withLength: NSStatusItem.variableLength)
        // 버튼 타이틀: 현재 입력 소스 기준 배지
        statusItem?.button?.title = shortBadge(for: currentInputSource)

        let menu = NSMenu()
        menu.addItem(withTitle: "현재 언어: \(currentInputSource)", action: nil, keyEquivalent: "")
        menu.addItem(.separator())
        menu.addItem(withTitle: "종료", action: #selector(quitApp), keyEquivalent: "q")
        statusItem?.menu = menu
    }
    
    private func updateStatusMenu() {
        let apply: () -> Void = { [weak self] in
            guard let self else { return }
            // 메뉴 1행: 현재 언어 전체 이름
            if let menu = self.statusItem?.menu, menu.numberOfItems > 0 {
                menu.item(at: 0)?.title = "현재 언어: \(self.currentInputSource)"
            }
            // 버튼 배지: A 또는 한
            self.statusItem?.button?.title = self.shortBadge(for: self.currentInputSource)
        }
        if Thread.isMainThread { apply() } else { DispatchQueue.main.async(execute: apply) }
    }
    
    @objc private func quitApp() {
        cleanup()
        NSApplication.shared.terminate(nil)
    }

    // -------------------- Input Source Change --------------------
    private func handleInputSourceChange() {
        let newInputSource = getCurrentInputSource()
        guard newInputSource != currentInputSource else { return }

        NSLog("입력 언어 변경: \(currentInputSource) → \(newInputSource)")
        currentInputSource = newInputSource
        updateStatusMenu()

        if notifyAlways {
            // ✅ 앱 전환 여부와 무관하게 항상 HUD 표시
            HUDManager.shared.show(text: newInputSource)
        } else {
            // ✅ 예전 방식: 앱 전환 직후(임계값 내)일 때만 표시
            let deltaMs = Date().timeIntervalSince(lastActivatedAt) * 1000
            if deltaMs >= 0 && deltaMs <= appSwitchThresholdMs {
                HUDManager.shared.show(text: newInputSource)
            }
        }
    }
    // -------------------- 안전 브리지 (포인터 브리지 대응) --------------------
    @inline(__always)
    private func TISPropertyAsString(_ src: TISInputSource, _ key: CFString) -> String? {
        guard let rawPtr = TISGetInputSourceProperty(src, key) as UnsafeMutableRawPointer? else {
            return nil
        }
        let cfAny = Unmanaged<CFTypeRef>.fromOpaque(rawPtr).takeUnretainedValue()
        guard CFGetTypeID(cfAny) == CFStringGetTypeID() else {
            NSLog("TISPropertyAsString: unexpected CFTypeID \(CFGetTypeID(cfAny)) for key \(key)")
            return nil
        }
        return (cfAny as! CFString) as String
    }

    private func getCurrentInputSource() -> String {
        guard let src = TISCopyCurrentKeyboardInputSource()?.takeRetainedValue() else {
            return "알 수 없음"
        }

        if let name = TISPropertyAsString(src, kTISPropertyLocalizedName) {
            return name
        }
        if let id = TISPropertyAsString(src, kTISPropertyInputSourceID) {
            if id.contains("Korean") || id.contains("Hangul") || id.contains("2SetKorean") { return "한국어" }
            if id.contains("ABC") || id.contains("US") { return "영어" }
            return id
        }
        return "알 수 없음"
    }

    // -------------------- Cleanup --------------------
    private func cleanup() {
        if let appObserver {
            NSWorkspace.shared.notificationCenter.removeObserver(appObserver)
            self.appObserver = nil
        }
        if let token = distToken {
            DistributedNotificationCenter.default().removeObserver(token)
            distToken = nil
        }
        HUDManager.shared.invalidate()
        if let s = statusItem {
            NSStatusBar.system.removeStatusItem(s)
            statusItem = nil
        }
    }
}
